/*
 * Copyright (c) 2003, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test
 * @bug 4851638 8301396
 * @key randomness
 * @library /test/lib
 * @build jdk.test.lib.RandomFactory
 * @build Tests
 * @build FdlibmTranslit
 * @build Expm1Tests
 * @run main Expm1Tests
 * @summary Tests for StrictMath.expm1
 */
import jdk.test.lib.RandomFactory;

/**
 * The tests in ../Math/Expm1Tests.java test properties that should
 * hold for any expm1 implementation, including the FDLIBM-based one
 * required for StrictMath.expm1.  Therefore, the test cases in
 * ../Math/Expm1Tests.java are run against both the Math and
 * StrictMath versions of expm1.  The role of this test is to verify
 * that the FDLIBM expm1 algorithm is being used by running golden
 * file tests on values that may vary from one conforming expm1
 * implementation to another.
 */

public class Expm1Tests {
    private Expm1Tests(){}

    public static void main(String... args) {
        int failures = 0;

        failures += testAgainstTranslit();
        failures += testExpm1();

        if (failures > 0) {
            System.err.println("Testing expm1 incurred "
                               + failures + " failures.");
            throw new RuntimeException();
        }
    }

    // Initialize shared random number generator
    private static java.util.Random random = RandomFactory.getRandom();

    /**
     * Test StrictMath.expm1 against transliteration port of expm1.
     */
    private static int testAgainstTranslit() {
        int failures = 0;
        double x;

        // Test just above subnormal threshold...
        x = Double.MIN_NORMAL;
        failures += testRange(x, Math.ulp(x), 1000);

        // ... and just below subnormal threshold ...
        x = Math.nextDown(Double.MIN_NORMAL);
        failures += testRange(x, -Math.ulp(x), 1000);

        // ... and near 1.0 ...
        failures += testRangeMidpoint(1.0, Math.ulp(x), 2000);
        // (Note: probes every-other value less than 1.0 due to
        // change in the size of an ulp at 1.0.

        // Probe near decision points in the FDLIBM algorithm.
        double LN2 = StrictMath.log(2.0);
        double[] decisionPoints = {
             7.09782712893383973096e+02, // overflow threshold
             56.0 * LN2,
            -56.0 * LN2,
             0.5 * LN2,
            -0.5 * LN2,
             1.5 * LN2,
            -1.5 * LN2,
             0x1.0p-54,
            -0x1.0p-54,
        };

        for (double testPoint : decisionPoints) {
            failures += testRangeMidpoint(testPoint, Math.ulp(testPoint), 1000);
        }

        x = Tests.createRandomDouble(random);

        // Make the increment twice the ulp value in case the random
        // value is near an exponent threshold. Don't worry about test
        // elements overflowing to infinity if the starting value is
        // near Double.MAX_VALUE.
        failures += testRange(x, 2.0 * Math.ulp(x), 1000);

        return failures;
    }

    private static int testRange(double start, double increment, int count) {
        int failures = 0;
        double x = start;
        for (int i = 0; i < count; i++, x += increment) {
            failures += testExpm1Case(x, FdlibmTranslit.expm1(x));
        }
        return failures;
    }

    private static int testRangeMidpoint(double midpoint, double increment, int count) {
        int failures = 0;
        double x = midpoint - increment*(count / 2) ;
        for (int i = 0; i < count; i++, x += increment) {
            failures += testExpm1Case(x, FdlibmTranslit.expm1(x));
        }
        return failures;
    }

    private static int testExpm1Case(double input, double expected) {
        return Tests.test("StrictMath.expm1(double)", input,
                          StrictMath::expm1, expected);
    }

    private static int testExpm1() {
        int failures = 0;

        // Test cases in the range [-36.75, 710]
        double [][] testCases = {
            {-0x1.580000008c619p3,      -0x1.fffd3069586f6p-1},
            {-0x1.380000008c721p3,      -0x1.fff85bf4a6e98p-1},
            {-0x1.180000008c9fap3,      -0x1.ffeb3aeb95524p-1},
            {-0x1.f0000001197ccp2,      -0x1.ffc78aadc116ap-1},
            {-0x1.b0000001197e7p2,      -0x1.ff6687cca710bp-1},
            {-0x1.70000001197f6p2,      -0x1.fe5ed3992a519p-1},
            {-0x1.30000001198p2,        -0x1.fb9201482bdfap-1},
            {-0x1.e000000233006p1,      -0x1.f3f57b658d6fbp-1},
            {-0x1.6000000233012p1,      -0x1.df44d8ee30b76p-1},
            {-0x1.c000000466028p0,      -0x1.a7071a097ed5ep-1},
            {-0x1.80000008cc052p-1,     -0x1.0e25f8a4a95b7p-1},
            {0x1.ffffffdccfeb7p-3,      0x1.22d78ef909144p-2},
            {0x1.3ffffffb99fd7p0,       0x1.3ec38ed3629a6p1},
            {0x1.1ffffffdccfebp1,       0x1.0f9b882a107edp3},
            {0x1.9ffffffdccfebp1,       0x1.8ca53b70fa11bp4},
            {0x1.0ffffffee67f6p2,       0x1.146bf132050c5p6},
            {0x1.4ffffffee67f5p2,       0x1.7b21ede9f0bdap7},
            {0x1.8ffffffee67f5p2,       0x1.0281a438aa523p9},
            {0x1.cffffffee67f5p2,       0x1.5fc6b5cf322c4p10},
            {0x1.07ffffff733fap3,       0x1.de5406b276b92p11},
            {0x1.27ffffff733fap3,       0x1.451c8690d1567p13},
            {0x1.47ffffff733fap3,       0x1.b9e62ae5924dfp14},
            {0x1.67ffffff733fap3,       0x1.2c4eeb7089cp16},
            {0x1.87ffffff733fap3,       0x1.982a24f2ab78ap17},
            {0x1.a7ffffff733fap3,       0x1.1560a14319349p19},
            {0x1.c7ffffff733fap3,       0x1.78fed772b40f2p20},
            {0x1.e7ffffff733fap3,       0x1.0031f18ee602fp22},
            {0x1.03ffffffb99fdp4,       0x1.5c348d8118f26p23},
            {0x1.13ffffffb99fdp4,       0x1.d942943e22d74p24},
            {0x1.23ffffffb99fcp4,       0x1.419d1309466ep26},
            {0x1.33ffffffb99fcp4,       0x1.b51e403430afep27},
            {0x1.43ffffffb99fcp4,       0x1.290d76c47bd4cp29},
            {0x1.53ffffffb99fcp4,       0x1.93bc8061146dp30},
            {0x1.63ffffffb99fbp4,       0x1.125e0665544a5p32},
            {0x1.73ffffffb99fbp4,       0x1.74e75f9de5d7cp33},
            {0x1.83ffffffb99fbp4,       0x1.fad42d3f28732p34},
            {0x1.93ffffffb99fbp4,       0x1.586d071cb8f87p36},
            {0x1.a3ffffffb99fbp4,       0x1.d41f91d0b4e6ep37},
            {0x1.b3ffffffb99fbp4,       0x1.3e1f6e5bc0242p39},
            {0x1.c3ffffffb99fbp4,       0x1.b05fa9aebfa64p40},
            {0x1.d3ffffffb99fbp4,       0x1.25d410cc90a38p42},
            {0x1.e3ffffffb99fbp4,       0x1.8f5aab33aa6c6p43},
            {0x1.f3ffffffb99fbp4,       0x1.0f63a91bc9797p45},
            {0x1.01ffffffdccfep5,       0x1.70db367c88b28p46},
            {0x1.09ffffffdccfep5,       0x1.f553e36d2975fp47},
            {0x1.11ffffffdccfep5,       0x1.54afff2230e99p49},
            {0x1.19ffffffdccfep5,       0x1.cf0ad451f1e9fp50},
            {0x1.21ffffffdccfep5,       0x1.3aab7c88ef991p52},
            {0x1.29ffffffdccfep5,       0x1.abae41ecccd22p53},
            {0x1.31ffffffdccfep5,       0x1.22a3a0462535fp55},
            {0x1.39ffffffdccfdp5,       0x1.8b050329f95c8p56},
            {0x1.41ffffffdccfdp5,       0x1.0c719224d80a2p58},
            {0x1.49ffffffdccfdp5,       0x1.6cda4c755ea56p59},
            {0x1.51ffffffdccfdp5,       0x1.efe2e2b6ad6ebp60},
            {0x1.59ffffffdccfdp5,       0x1.50fd5a6337c61p62},
            {0x1.61ffffffdccfdp5,       0x1.ca043518d78acp63},
            {0x1.69ffffffdccfdp5,       0x1.374122dd2fbdbp65},
            {0x1.71ffffffdccfdp5,       0x1.a709e46cc671ep66},
            {0x1.79ffffffdccfdp5,       0x1.1f7c0c5482bf3p68},
            {0x1.81ffffffdccfdp5,       0x1.86bb667297515p69},
            {0x1.89ffffffdccfcp5,       0x1.0987aa8375abcp71},
            {0x1.91ffffffdccfcp5,       0x1.68e48248f27ddp72},
            {0x1.99ffffffdccfcp5,       0x1.ea8100a2e27e9p73},
            {0x1.a1ffffffdccfcp5,       0x1.4d54fc02d9352p75},
            {0x1.a9ffffffdccfcp5,       0x1.c50b8ceab6ad1p76},
            {0x1.b1ffffffdccfcp5,       0x1.33e046afc7062p78},
            {0x1.b9ffffffdccfcp5,       0x1.a2726cf2e78e3p79},
            {0x1.c1ffffffdccfcp5,       0x1.1c5d3c581edf2p81},
            {0x1.c9ffffffdccfcp5,       0x1.827db3961daecp82},
            {0x1.d1ffffffdccfbp5,       0x1.06a5db797b4b2p84},
            {0x1.d9ffffffdccfbp5,       0x1.64f9b90e23fb4p85},
            {0x1.e1ffffffdccfbp5,       0x1.e52e132ebafe2p86},
            {0x1.e9ffffffdccfbp5,       0x1.49b6c774442efp88},
            {0x1.f1ffffffdccfbp5,       0x1.c020b4f9d926cp89},
            {0x1.f9ffffffdccfbp5,       0x1.3088cda20d465p91},
            {0x1.00ffffffee67ep6,       0x1.9de7b7a818186p92},
            {0x1.04ffffffee67ep6,       0x1.194717f5da259p94},
            {0x1.08ffffffee67ep6,       0x1.7e4bc97a2360dp95},
            {0x1.0cffffffee67ep6,       0x1.03cc0e87f367bp97},
            {0x1.10ffffffee67ep6,       0x1.6119d231b67f5p98},
            {0x1.14ffffffee67ep6,       0x1.dfe9f0cbe5942p99},
            {0x1.18ffffffee67ep6,       0x1.4622a079fc2a6p101},
            {0x1.1cffffffee67ep6,       0x1.bb4386e45ae94p102},
            {0x1.20ffffffee67ep6,       0x1.2d3a9d9e9fe6p104},
            {0x1.24ffffffee67ep6,       0x1.9969a118d6261p105},
            {0x1.28ffffffee67ep6,       0x1.1639871642331p107},
            {0x1.2cffffffee67ep6,       0x1.7a2587603a84bp108},
            {0x1.30ffffffee67ep6,       0x1.00fa2d6e6a76ep110},
            {0x1.34ffffffee67ep6,       0x1.5d44af7562574p111},
            {0x1.38ffffffee67ep6,       0x1.dab4705f88c02p112},
            {0x1.3cffffffee67ep6,       0x1.42986b24fc9dcp114},
            {0x1.40ffffffee67ep6,       0x1.b673dcb2fe519p115},
            {0x1.44ffffffee67ep6,       0x1.29f59cd896383p117},
            {0x1.48ffffffee67ep6,       0x1.94f806342143cp118},
            {0x1.4cffffffee67ep6,       0x1.133471e4d5b38p120},
            {0x1.50ffffffee67ep6,       0x1.760acce4f0e03p121},
            {0x1.54ffffffee67ep6,       0x1.fc604454828ddp122},
            {0x1.58ffffffee67ep6,       0x1.597a32eee8c46p124},
            {0x1.5cffffffee67ep6,       0x1.d58d694102246p125},
            {0x1.60ffffffee67ep6,       0x1.3f180bd3df0d2p127},
            {0x1.64ffffffee67ep6,       0x1.b1b190d803f07p128},
            {0x1.68ffffffee67ep6,       0x1.26b9b1cab82dap130},
            {0x1.6cffffffee67ep6,       0x1.9092c44a68bc1p131},
            {0x1.70ffffffee67ep6,       0x1.1037c0cf4a5a6p133},
            {0x1.74ffffffee67ep6,       0x1.71fb79fed30fbp134},
            {0x1.78ffffffee67ep6,       0x1.f6dbadec024eep135},
            {0x1.7cffffffee67ep6,       0x1.55ba3f072a6dbp137},
            {0x1.80ffffffee67ep6,       0x1.d074b338a9163p138},
            {0x1.84ffffffee67ep6,       0x1.3ba167320351ap140},
            {0x1.88ffffffee67ep6,       0x1.acfc7e2e0558bp141},
            {0x1.8cffffffee67ep6,       0x1.2386c336b7163p143},
            {0x1.90ffffffee67ep6,       0x1.8c39b90c7cdap144},
            {0x1.94ffffffee67ep6,       0x1.0d435c84d4e66p146},
            {0x1.98ffffffee67ep6,       0x1.6df76efd7275ep147},
            {0x1.9cffffffee67ep6,       0x1.f1666c9163f86p148},
            {0x1.a0ffffffee67ep6,       0x1.5204b679406b7p150},
            {0x1.a4ffffffee67ep6,       0x1.cb6a267e94b7ap151},
            {0x1.a8ffffffee67ep6,       0x1.38346236ba483p153},
            {0x1.acffffffee67ep6,       0x1.a8547ff6d3311p154},
            {0x1.b0ffffffee67ep6,       0x1.205cb8246899ap156},
            {0x1.b4ffffffee67ep6,       0x1.87ecc28a831c1p157},
            {0x1.b8ffffffee67ep6,       0x1.0a572df57323ep159},
            {0x1.bcffffffee67ep6,       0x1.69fe8c886de24p160},
            {0x1.c0ffffffee67ep6,       0x1.ec0055aa644acp161},
            {0x1.c4ffffffee67ep6,       0x1.4e597c5197d13p163},
            {0x1.c8ffffffee67ep6,       0x1.c66d9bb965746p164},
            {0x1.ccffffffee67ep6,       0x1.34d0e22472ce2p166},
            {0x1.d0ffffffee67ep6,       0x1.a3b971da5668ap167},
            {0x1.d4ffffffee67ep6,       0x1.1d3b77e103d1ap169},
            {0x1.d8ffffffee67ep6,       0x1.83abbf32ed4f6p170},
            {0x1.dcffffffee67ep6,       0x1.07731e5137e95p172},
            {0x1.e0ffffffee67ep6,       0x1.6610b39e7ce7p173},
            {0x1.e4ffffffee67ep6,       0x1.e6a93f132076cp174},
            {0x1.e8ffffffee67ep6,       0x1.4ab873ed0fb2cp176},
            {0x1.ecffffffee67ep6,       0x1.c17eebfd11debp177},
            {0x1.f0ffffffee67ep6,       0x1.3176cc87e9082p179},
            {0x1.f4ffffffee67ep6,       0x1.9f2b2fe57487ap180},
            {0x1.f8ffffffee67ep6,       0x1.1a22e9fe60816p182},
            {0x1.fcffffffee67ep6,       0x1.7f768dd1738aap183},
            {0x1.007ffffff733fp7,       0x1.049717079907bp185},
            {0x1.027ffffff733fp7,       0x1.622dc5947dd63p186},
            {0x1.047ffffff733fp7,       0x1.e160ff1ccd30cp187},
            {0x1.067ffffff733fp7,       0x1.472180f8199d7p189},
            {0x1.087ffffff733fp7,       0x1.bc9df0c9b6e0fp190},
            {0x1.0a7ffffff733fp7,       0x1.2e26073757ed6p192},
            {0x1.0c7ffffff733fp7,       0x1.9aa99688f714bp193},
            {0x1.0e7ffffff733fp7,       0x1.1712f6523864dp195},
            {0x1.107ffffff733fp7,       0x1.7b4d0d8e11012p196},
            {0x1.127ffffff733fp7,       0x1.01c301c6bf29cp198},
            {0x1.147ffffff733fp7,       0x1.5e55a41486608p199},
            {0x1.167ffffff733fp7,       0x1.dc276c8c7156p200},
            {0x1.187ffffff733fp7,       0x1.4394876ddc7dap202},
            {0x1.1a7ffffff733fp7,       0x1.b7ca840a6b3bap203},
            {0x1.1c7ffffff733fp7,       0x1.2ade7851ad0fep205},
            {0x1.1e7ffffff733fp7,       0x1.963482987606p206},
            {0x1.207ffffff733fp7,       0x1.140b84f56a91bp208},
            {0x1.227ffffff733fp7,       0x1.772f1dec03a66p209},
            {0x1.247ffffff733fp7,       0x1.fded90f5af3ap210},
            {0x1.267ffffff733fp7,       0x1.5a88311cf6cbdp212},
            {0x1.287ffffff733fp7,       0x1.d6fc5e99a419dp213},
            {0x1.2a7ffffff733fp7,       0x1.40116b9759ebap215},
            {0x1.2c7ffffff733fp7,       0x1.b304801416466p216},
            {0x1.2e7ffffff733fp7,       0x1.27a0063dbe9cep218},
            {0x1.307ffffff733fp7,       0x1.91cbd14945353p219},
            {0x1.327ffffff733fp7,       0x1.110c7e4340e4fp221},
            {0x1.347ffffff733fp7,       0x1.731c9ec8ce996p222},
            {0x1.367ffffff733fp7,       0x1.f864aa9acffbdp223},
            {0x1.387ffffff733fp7,       0x1.56c54eff8fbcdp225},
            {0x1.3a7ffffff733fp7,       0x1.d1dfaced4eb1p226},
            {0x1.3c7ffffff733fp7,       0x1.3c98120a95d78p228},
            {0x1.3e7ffffff733fp7,       0x1.ae4bbfa449eaap229},
            {0x1.407ffffff733fp7,       0x1.246a97a9838dcp231},
            {0x1.427ffffff733fp7,       0x1.8d6f603164cebp232},
            {0x1.447ffffff733fp7,       0x1.0e15cad8b775ep234},
            {0x1.467ffffff733fp7,       0x1.6f15705b3f514p235},
            {0x1.487ffffff733fp7,       0x1.f2eb25494787dp236},
            {0x1.4a7ffffff733fp7,       0x1.530ce0608a8acp238},
            {0x1.4c7ffffff733fp7,       0x1.ccd12fa07172p239},
            {0x1.4e7ffffff733fp7,       0x1.39285fa9c08e7p241},
            {0x1.507ffffff733fp7,       0x1.a9a01de01fd02p242},
            {0x1.527ffffff733fp7,       0x1.213e13894e05p244},
            {0x1.547ffffff733fp7,       0x1.891f0d4674b33p245},
            {0x1.567ffffff733fp7,       0x1.0b275393c60b2p247},
            {0x1.587ffffff733fp7,       0x1.6b1973327581dp248},
            {0x1.5a7ffffff733fp7,       0x1.ed80d645874cfp249},
            {0x1.5c7ffffff733fp7,       0x1.4f5ec835b4172p251},
            {0x1.5e7ffffff733fp7,       0x1.c7d0bf3aec5fcp252},
            {0x1.607ffffff733fp7,       0x1.35c239a263125p254},
            {0x1.627ffffff733fp7,       0x1.a501765319bp255},
            {0x1.647ffffff733fp7,       0x1.1e1a611707cfbp257},
            {0x1.667ffffff733fp7,       0x1.84dab6dcaac3cp258},
            {0x1.687ffffff733fp7,       0x1.08410192ab89ep260},
            {0x1.6a7ffffff733fp7,       0x1.67288834edb2fp261},
            {0x1.6c7ffffff733fp7,       0x1.e825934abdad9p262},
            {0x1.6e7ffffff733fp7,       0x1.4bbae9c58a222p264},
            {0x1.707ffffff733fp7,       0x1.c2de34b24b161p265},
            {0x1.727ffffff733fp7,       0x1.3265856c8dbdcp267},
            {0x1.747ffffff733fp7,       0x1.a06fa4ee04c8ep268},
            {0x1.767ffffff733fp7,       0x1.1aff67d171068p270},
            {0x1.787ffffff733fp7,       0x1.80a23ba5cc0fep271},
            {0x1.7a7ffffff733fp7,       0x1.0562be333b5b6p273},
            {0x1.7c7ffffff733fp7,       0x1.6342909f8e806p274},
            {0x1.7e7ffffff733fp7,       0x1.e2d932898c11cp275},
            {0x1.807ffffff733ep7,       0x1.482128a65b0aap277},
            {0x1.827ffffff733ep7,       0x1.bdf9696894p278},
            {0x1.847ffffff733ep7,       0x1.2f1228ca0924p280},
            {0x1.867ffffff733ep7,       0x1.9bea8605e05b8p281},
            {0x1.887ffffff733ep7,       0x1.17ed0f7b60befp283},
            {0x1.8a7ffffff733ep7,       0x1.7c757ab028d2cp284},
            {0x1.8c7ffffff733ep7,       0x1.028c73122cbaap286},
            {0x1.8e7ffffff733ep7,       0x1.5f676e04b872p287},
            {0x1.907ffffff733ep7,       0x1.dd9b8aa6c07f4p288},
            {0x1.927ffffff733ep7,       0x1.449168bd6830bp290},
            {0x1.947ffffff733ep7,       0x1.b922372b1b22fp291},
            {0x1.967ffffff733ep7,       0x1.2bc809c589606p293},
            {0x1.987ffffff733ep7,       0x1.9771f652c776ap294},
            {0x1.9a7ffffff733ep7,       0x1.14e3401b07fc7p296},
            {0x1.9c7ffffff733ep7,       0x1.785453659b7d4p297},
            {0x1.9e7ffffff733ep7,       0x1.ff7c1414d829fp298},
            {0x1.a07ffffff733ep7,       0x1.5b97024b58a63p300},
            {0x1.a27ffffff733ep7,       0x1.d86c72ba13072p301},
            {0x1.a47ffffff733ep7,       0x1.410b8e3e0a59p303},
            {0x1.a67ffffff733ep7,       0x1.b458783157a34p304},
            {0x1.a87ffffff733ep7,       0x1.28870eb1e33efp306},
            {0x1.aa7ffffff733ep7,       0x1.9305d2eedd47p307},
            {0x1.ac7ffffff733ep7,       0x1.11e1e1f93656cp309},
            {0x1.ae7ffffff733ep7,       0x1.743ea58a8a142p310},
            {0x1.b07ffffff733ep7,       0x1.f9eeda68dbff6p311},
            {0x1.b27ffffff733ep7,       0x1.57d12fadfda18p313},
            {0x1.b47ffffff733ep7,       0x1.d34bc24ce61e7p314},
            {0x1.b67ffffff733ep7,       0x1.3d8f7da8d8d95p316},
            {0x1.b87ffffff733ep7,       0x1.af9c071bbd116p317},
            {0x1.ba7ffffff733ep7,       0x1.254f1e2943f7fp319},
            {0x1.bc7ffffff733ep7,       0x1.8ea5f9553ce5ep320},
            {0x1.be7ffffff733ep7,       0x1.0ee8dda0a100cp322},
            {0x1.c07ffffff733ep7,       0x1.7034513ceac7dp323},
            {0x1.c27ffffff733ep7,       0x1.f4710dcb08bdcp324},
            {0x1.c47ffffff733ep7,       0x1.5415d8b9ef19cp326},
            {0x1.c67ffffff733ep7,       0x1.ce3951590b045p327},
            {0x1.c87ffffff733ep7,       0x1.3a1d1bcad2ec4p329},
            {0x1.ca7ffffff733ep7,       0x1.aaecbef297a4ap330},
            {0x1.cc7ffffff733ep7,       0x1.22201f0c6ae88p332},
            {0x1.ce7ffffff733ep7,       0x1.8a524760ebdc1p333},
            {0x1.d07ffffff733ep7,       0x1.0bf81bdd2ba52p335},
            {0x1.d27ffffff733ep7,       0x1.6c3536f34b0c2p336},
            {0x1.d47ffffff733ep7,       0x1.ef02835e6a7d8p337},
            {0x1.d67ffffff733ep7,       0x1.5064e04e480fep339},
            {0x1.d87ffffff733ep7,       0x1.c934f847894a8p340},
            {0x1.da7ffffff733ep7,       0x1.36b44dbc8b633p342},
            {0x1.dc7ffffff733ep7,       0x1.a64a7b24ebae2p343},
            {0x1.de7ffffff733ep7,       0x1.1ef9f881e57b8p345},
            {0x1.e07ffffff733ep7,       0x1.860a9b4bcf9dfp346},
            {0x1.e27ffffff733ep7,       0x1.090f85bb33493p348},
            {0x1.e47ffffff733ep7,       0x1.6841377bd96e9p349},
            {0x1.e67ffffff733ep7,       0x1.e9a310bd2715dp350},
            {0x1.e87ffffff733ep7,       0x1.4cbe299b1372dp352},
            {0x1.ea7ffffff733ep7,       0x1.c43e8fef69d0dp353},
            {0x1.ec7ffffff733ep7,       0x1.3354f8e156a72p355},
            {0x1.ee7ffffff733ep7,       0x1.a1b51787582dfp356},
            {0x1.f07ffffff733ep7,       0x1.1bdc91f54d333p358},
            {0x1.f27ffffff733ep7,       0x1.81ced3ada5ec9p359},
            {0x1.f47ffffff733ep7,       0x1.062f0486db268p361},
            {0x1.f67ffffff733ep7,       0x1.645833fb72196p362},
            {0x1.f87ffffff733ep7,       0x1.e4528bf7332d2p363},
            {0x1.fa7ffffff733ep7,       0x1.492198206b3aep365},
            {0x1.fc7ffffff733ep7,       0x1.bf55f194851b5p366},
            {0x1.fe7ffffff733ep7,       0x1.2fff02e67affep368},
            {0x1.003ffffffb99fp8,       0x1.9d2c7052fc80ep369},
            {0x1.013ffffffb99fp8,       0x1.18c7d31687cep371},
            {0x1.023ffffffb99fp8,       0x1.7d9ecf7b00132p372},
            {0x1.033ffffffb99fp8,       0x1.035681cb5b766p374},
            {0x1.043ffffffb99fp8,       0x1.607a0decadfaep375},
            {0x1.053ffffffb99fp8,       0x1.df10cb910ae14p376},
            {0x1.063ffffffb99fp8,       0x1.458f0fad99f57p378},
            {0x1.073ffffffb99fp8,       0x1.ba7af6e654fa2p379},
            {0x1.083ffffffb99fp8,       0x1.2cb251c2631d6p381},
            {0x1.093ffffffb99fp8,       0x1.98b06224611f2p382},
            {0x1.0a3ffffffb99fp8,       0x1.15bba3d909807p384},
            {0x1.0b3ffffffb99fp8,       0x1.797a6e0440f8fp385},
            {0x1.0c3ffffffb99fp8,       0x1.0085e752522ap387},
            {0x1.0d3ffffffb99fp8,       0x1.5ca6a71ef489ep388},
            {0x1.0e3ffffffb99fp8,       0x1.d9dda6826dfc7p389},
            {0x1.0f3ffffffb99fp8,       0x1.420674603ebb2p391},
            {0x1.103ffffffb99fp8,       0x1.b5ad79fec979p392},
            {0x1.113ffffffb99fp8,       0x1.296ecbb3d2db7p394},
            {0x1.123ffffffb99fp8,       0x1.9440c9fa6362ap395},
            {0x1.133ffffffb99fp8,       0x1.12b7ec73193e4p397},
            {0x1.143ffffffb99fp8,       0x1.75618ef49dffep398},
            {0x1.153ffffffb99fp8,       0x1.fb7a3e462b12ap399},
            {0x1.163ffffffb99fp8,       0x1.58dde1b590206p401},
            {0x1.173ffffffb99fp8,       0x1.d4b8f4351faecp402},
            {0x1.183ffffffb99fp8,       0x1.3e87aaa373892p404},
            {0x1.193ffffffb99fp8,       0x1.b0ed5561210d9p405},
            {0x1.1a3ffffffb99fp8,       0x1.263457411e3a5p407},
            {0x1.1b3ffffffb99fp8,       0x1.8fdd8535244cdp408},
            {0x1.1c3ffffffb99fp8,       0x1.0fbc955d170d1p410},
            {0x1.1d3ffffffb99fp8,       0x1.7154125122a86p411},
            {0x1.1e3ffffffb99fp8,       0x1.f5f8270411675p412},
            {0x1.1f3ffffffb99fp8,       0x1.551fa026c4e4cp414},
            {0x1.203ffffffb99fp8,       0x1.cfa28c83a9c15p415},
            {0x1.213ffffffb99fp8,       0x1.3b12972ef5f7ap417},
            {0x1.223ffffffb99fp8,       0x1.ac3a63f8c3fc9p418},
            {0x1.233ffffffb99fp8,       0x1.2302db376285dp420},
            {0x1.243ffffffb99fp8,       0x1.8b867194fa443p421},
            {0x1.253ffffffb99fp8,       0x1.0cc98750c45aep423},
            {0x1.263ffffffb99fp8,       0x1.6d51d877b6f1bp424},
            {0x1.273ffffffb99fp8,       0x1.f0855ddfe3faap425},
            {0x1.283ffffffb99fp8,       0x1.516bc53aea3aap427},
            {0x1.293ffffffb99fp8,       0x1.ca9a47b823396p428},
            {0x1.2a3ffffffb99fp8,       0x1.37a71f0652462p430},
            {0x1.2b3ffffffb99fp8,       0x1.a794811822eb3p431},
            {0x1.2c3ffffffb99fp8,       0x1.1fda3ea9c1a4fp433},
            {0x1.2d3ffffffb99fp8,       0x1.873b6d3965cp434},
            {0x1.2e3ffffffb99fp8,       0x1.09deab488e539p436},
            {0x1.2f3ffffffb99fp8,       0x1.695ac21e2870ep437},
            {0x1.303ffffffb99fp8,       0x1.eb21b852a647cp438},
            {0x1.313ffffffb99fp8,       0x1.4dc2340b86b4cp440},
            {0x1.323ffffffb99fp8,       0x1.c59ffe8afa62cp441},
            {0x1.333ffffffb99fp8,       0x1.3445277810b78p443},
            {0x1.343ffffffb99fp8,       0x1.a2fb88779894fp444},
            {0x1.353ffffffb99fp8,       0x1.1cba68f09f8c5p446},
            {0x1.363ffffffb99fp8,       0x1.82fc56a008db6p447},
            {0x1.373ffffffb99fp8,       0x1.06fbea7eda2dfp449},
            {0x1.383ffffffb99fp8,       0x1.656eb0513614ep450},
            {0x1.393ffffffb99fp8,       0x1.e5cd0c4b86aaap451},
            {0x1.3a3ffffffb99fp8,       0x1.4a22d0026e7ffp453},
            {0x1.3b3ffffffb99fp8,       0x1.c0b38a21c236cp454},
            {0x1.3c3ffffffb99fp8,       0x1.30ec961ce5379p456},
            {0x1.3d3ffffffb99fp8,       0x1.9e6f56344e9dp457},
            {0x1.3e3ffffffb99fp8,       0x1.19a341a8e1ccbp459},
            {0x1.3f3ffffffb99fp8,       0x1.7ec90ca3a1c6dp460},
            {0x1.403ffffffb99fp8,       0x1.04212e6d536d6p462},
            {0x1.413ffffffb99fp8,       0x1.618d84739e91cp463},
            {0x1.423ffffffb99fp8,       0x1.e087302e9607ap464},
            {0x1.433ffffffb99fp8,       0x1.468d7cd8e4417p466},
            {0x1.443ffffffb99fp8,       0x1.bbd4c40e0317cp467},
            {0x1.453ffffffb99fp8,       0x1.2d9d50d6e1436p469},
            {0x1.463ffffffb99fp8,       0x1.99efc6cf25729p470},
            {0x1.473ffffffb99fp8,       0x1.1694b0b33138ap472},
            {0x1.483ffffffb99fp8,       0x1.7aa16e7b0810dp473},
            {0x1.493ffffffb99fp8,       0x1.014e60cc3c10bp475},
            {0x1.4a3ffffffb99fp8,       0x1.5db7203d316b3p476},
            {0x1.4b3ffffffb99fp8,       0x1.db4ffad383047p477},
            {0x1.4c3ffffffb99fp8,       0x1.43021e96bc60ep479},
            {0x1.4d3ffffffb99fp8,       0x1.b703864c0ed74p480},
            {0x1.4e3ffffffb99fp8,       0x1.2a573dd0a80e6p482},
            {0x1.4f3ffffffb99fp8,       0x1.957cb72b9f3f6p483},
            {0x1.503ffffffb99fp8,       0x1.138e9e333d9afp485},
            {0x1.513ffffffb99fp8,       0x1.76855bb82cbcep486},
            {0x1.523ffffffb99fp8,       0x1.fd06d7237d52bp487},
            {0x1.533ffffffb99fp8,       0x1.59eb65b9e296ap489},
            {0x1.543ffffffb99fp8,       0x1.d627438458c6ap490},
            {0x1.553ffffffb99fp8,       0x1.3f80999182b7ap492},
            {0x1.563ffffffb99fp8,       0x1.b23fab41d7fcdp493},
            {0x1.573ffffffb99fp8,       0x1.271a437ca4dd9p495},
            {0x1.583ffffffb99fp8,       0x1.9116048ecdd82p496},
            {0x1.593ffffffb99fp8,       0x1.1090f28f03784p498},
            {0x1.5a3ffffffb99fp8,       0x1.7274b4471d222p499},
            {0x1.5b3ffffffb99fp8,       0x1.f78071e284acfp500},
            {0x1.5c3ffffffb99fp8,       0x1.562a3748e0ae8p502},
            {0x1.5d3ffffffb99fp8,       0x1.d10ce1fc412f2p503},
            {0x1.5e3ffffffb99fp8,       0x1.3c08d26ba29edp505},
            {0x1.5f3ffffffb99fp8,       0x1.ad890dbdcc46ap506},
            {0x1.603ffffffb99fp8,       0x1.23e648944393ap508},
            {0x1.613ffffffb99fp8,       0x1.8cbb8c9e43a63p509},
            {0x1.623ffffffb99fp8,       0x1.0d9b966e13d61p511},
            {0x1.633ffffffb99fp8,       0x1.6e6f586d0888dp512},
            {0x1.643ffffffb99fp8,       0x1.f20966b5813aep513},
            {0x1.653ffffffb99fp8,       0x1.5273779badaf2p515},
            {0x1.663ffffffb99fp8,       0x1.cc00ae664a89p516},
            {0x1.673ffffffb99fp8,       0x1.389aae1391554p518},
            {0x1.683ffffffb99fp8,       0x1.a8df88f5b2588p519},
            {0x1.693ffffffb99fp8,       0x1.20bb34172b66ap521},
            {0x1.6a3ffffffb99fp8,       0x1.886d2d5f07833p522},
            {0x1.6b3ffffffb99fp8,       0x1.0aae72b8de01ep524},
            {0x1.6c3ffffffb99fp8,       0x1.6a7528c7487fep525},
            {0x1.6d3ffffffb99fp8,       0x1.eca18af43a36cp526},
            {0x1.6e3ffffffb99fp8,       0x1.4ec709b53a35fp528},
            {0x1.6f3ffffffb99fp8,       0x1.c702815c30a11p529},
            {0x1.703ffffffb99fp8,       0x1.353611c2fab72p531},
            {0x1.713ffffffb99fp8,       0x1.a442f8858a925p532},
            {0x1.723ffffffb99fp8,       0x1.1d98ed4a7bba3p534},
            {0x1.733ffffffb99fp8,       0x1.842ac5348b7cep535},
            {0x1.743ffffffb99fp8,       0x1.07c97097fb529p537},
            {0x1.753ffffffb99fp8,       0x1.6686064a6be2ap538},
            {0x1.763ffffffb99fp8,       0x1.e748b46cfe14cp539},
            {0x1.773ffffffb99fp8,       0x1.4b24d0e9033c7p541},
            {0x1.783ffffffb99fp8,       0x1.c21233e5293dap542},
            {0x1.793ffffffb99fp8,       0x1.31dae2fdf0407p544},
            {0x1.7a3ffffffb99fp8,       0x1.9fb3386e7303dp545},
            {0x1.7b3ffffffb99fp8,       0x1.1a7f5bb80b183p547},
            {0x1.7c3ffffffb99fp8,       0x1.7ff432dfa67a9p548},
            {0x1.7d3ffffffb99fp8,       0x1.04ec79737cde7p550},
            {0x1.7e3ffffffb99fp8,       0x1.62a1d2414486ep551},
            {0x1.7f3ffffffb99fp8,       0x1.e1feb963592a3p552},
            {0x1.803ffffffb99fp8,       0x1.478cb0da3248p554},
            {0x1.813ffffffb99fp8,       0x1.bd2f9f74b3ecep555},
            {0x1.823ffffffb99fp8,       0x1.2e8907921a545p557},
            {0x1.833ffffffb99fp8,       0x1.9b3025158e763p558},
            {0x1.843ffffffb99fp8,       0x1.176e672da841p560},
            {0x1.853ffffffb99fp8,       0x1.7bc9557d90bb5p561},
            {0x1.863ffffffb99fp8,       0x1.021776f23b20ap563},
            {0x1.873ffffffb99fp8,       0x1.5ec86e4bf78ccp564},
            {0x1.883ffffffb99fp8,       0x1.dcc3708ecfe88p565},
            {0x1.893ffffffb99fp8,       0x1.43fe8d7ac0079p567},
            {0x1.8a3ffffffb99fp8,       0x1.b85a9de96d206p568},
            {0x1.8b3ffffffb99fp8,       0x1.2b406595ebc2fp570},
            {0x1.8c3ffffffb99fp8,       0x1.96b99b42ee81ep571},
            {0x1.8d3ffffffb99fp8,       0x1.1465f7bc5d4f3p573},
            {0x1.8e3ffffffb99fp8,       0x1.77aa0c86e3254p574},
            {0x1.8f3ffffffb99fp8,       0x1.fe94a5f24f127p575},
            {0x1.903ffffffb99fp8,       0x1.5af9bc5f10484p577},
            {0x1.913ffffffb99fp8,       0x1.d796b1199ca0dp578},
            {0x1.923ffffffb99fp8,       0x1.407a4b0a99581p580},
            {0x1.933ffffffb99fp8,       0x1.b393098be48c6p581},
            {0x1.943ffffffb99fp8,       0x1.2800e367d7873p583},
            {0x1.953ffffffb99fp8,       0x1.924f782080a9p584},
            {0x1.963ffffffb99fp8,       0x1.1165f5b7b4e77p586},
            {0x1.973ffffffb99fp8,       0x1.739637ce995dep587},
            {0x1.983ffffffb99fp8,       0x1.f909ef553fe1dp588},
            {0x1.993ffffffb99fp8,       0x1.57359ec295bd8p590},
            {0x1.9a3ffffffb99fp8,       0x1.d278529f70c72p591},
            {0x1.9b3ffffffb99fp8,       0x1.3cffce16c6a9fp593},
            {0x1.9c3ffffffb99fp8,       0x1.aed8bd0d76c2ep594},
            {0x1.9d3ffffffb99fp8,       0x1.24ca67ad88b68p596},
            {0x1.9e3ffffffb99fp8,       0x1.8df19938fe6d4p597},
            {0x1.9f3ffffffb99fp8,       0x1.0e6e49b5016fbp599},
            {0x1.a03ffffffb99fp8,       0x1.6f8db78116a48p600},
            {0x1.a13ffffffb99fp8,       0x1.f38e9ecb87da4p601},
            {0x1.a23ffffffb99fp8,       0x1.537bf81122a93p603},
            {0x1.a33ffffffb99fp8,       0x1.cd682d2c39ab4p604},
            {0x1.a43ffffffb99fp8,       0x1.398efb7895c02p606},
            {0x1.a53ffffffb99fp8,       0x1.aa2b938729ffap607},
            {0x1.a63ffffffb99fp8,       0x1.219cd9531c9cep609},
            {0x1.a73ffffffb99fp8,       0x1.899fdc76e059bp610},
            {0x1.a83ffffffb99fp8,       0x1.0b7edc8aa647ep612},
            {0x1.a93ffffffb99fp8,       0x1.6b906c232d65ep613},
            {0x1.aa3ffffffb99fp8,       0x1.ee22898b97fbep614},
            {0x1.ab3ffffffb99fp8,       0x1.4fccab37000cep616},
            {0x1.ac3ffffffb99fp8,       0x1.c866193ae89dbp617},
            {0x1.ad3ffffffb99fp8,       0x1.3627b854c5c27p619},
            {0x1.ae3ffffffb99fp8,       0x1.a58b68788e1e6p620},
            {0x1.af3ffffffb99fp8,       0x1.1e781f8a5efe5p622},
            {0x1.b03ffffffb99fp8,       0x1.855a202353f77p623},
            {0x1.b13ffffffb99fp8,       0x1.0897974f62fe1p625},
            {0x1.b23ffffffb99fp8,       0x1.679e369129826p626},
            {0x1.b33ffffffb99fp8,       0x1.e8c58542c521dp627},
            {0x1.b43ffffffb99fp8,       0x1.4c279b7142392p629},
            {0x1.b53ffffffb99fp8,       0x1.c371efb43e76ep630},
            {0x1.b63ffffffb99fp8,       0x1.32c9ea1ab59ccp632},
            {0x1.b73ffffffb99fp8,       0x1.a0f817c69fb0ap633},
            {0x1.b83ffffffb99fp8,       0x1.1b5c21ca08788p635},
            {0x1.b93ffffffb99fp8,       0x1.812042e534a64p636},
            {0x1.ba3ffffffb99fp8,       0x1.05b86359a079cp638},
            {0x1.bb3ffffffb99fp8,       0x1.63b6f7fddd3efp639},
            {0x1.bc3ffffffb99fp8,       0x1.e3776813fda8ep640},
            {0x1.bd3ffffffb99fp8,       0x1.488cac4ce84e7p642},
            {0x1.be3ffffffb99fp8,       0x1.be8b89ed9a77dp643},
            {0x1.bf3ffffffb99fp8,       0x1.2f75768394a5p645},
            {0x1.c03ffffffb99fp8,       0x1.9c717dbaae2b8p646},
            {0x1.c13ffffffb99fp8,       0x1.1848c7ccfefe3p648},
            {0x1.c23ffffffb99fp8,       0x1.7cf223c0074fbp649},
            {0x1.c33ffffffb99fp8,       0x1.02e12a3ec0173p651},
            {0x1.c43ffffffb99fp8,       0x1.5fda91f1b0d98p652},
            {0x1.c53ffffffb99fp8,       0x1.de38089682abp653},
            {0x1.c63ffffffb99fp8,       0x1.44fbc1a5fe2ddp655},
            {0x1.c73ffffffb99fp8,       0x1.b9b2c1a7cc7aap656},
            {0x1.c83ffffffb99fp8,       0x1.2c2a43919580dp658},
            {0x1.c93ffffffb99fp8,       0x1.97f7770145248p659},
            {0x1.ca3ffffffb99fp8,       0x1.153df9919867p661},
            {0x1.cb3ffffffb99fp8,       0x1.78cfa212f8edcp662},
            {0x1.cc3ffffffb99fp8,       0x1.0011d5d26caedp664},
            {0x1.cd3ffffffb99fp8,       0x1.5c08e649b4b94p665},
            {0x1.ce3ffffffb99fp8,       0x1.d9073dd4a4c7bp666},
            {0x1.cf3ffffffb99fp8,       0x1.4174bfa6c0d24p668},
            {0x1.d03ffffffb99fp8,       0x1.b4e7710dea691p669},
            {0x1.d13ffffffb99fp8,       0x1.28e8378f2345ap671},
            {0x1.d23ffffffb99fp8,       0x1.9389e0a91894fp672},
            {0x1.d33ffffffb99fp8,       0x1.123b9f58df0dap674},
            {0x1.d43ffffffb99fp8,       0x1.74b89d97dfdd3p675},
            {0x1.d53ffffffb99fp8,       0x1.fa94a04bdb05cp676},
            {0x1.d63ffffffb99fp8,       0x1.5841d736b633cp678},
            {0x1.d73ffffffb99fp8,       0x1.d3e4df4a846ddp679},
            {0x1.d83ffffffb99fp8,       0x1.3df78ac6c50cep681},
            {0x1.d93ffffffb99fp8,       0x1.b02972b428f19p682},
            {0x1.da3ffffffb99fp8,       0x1.25af390e18cbep684},
            {0x1.db3ffffffb99fp8,       0x1.8f289821f41b9p685},
            {0x1.dc3ffffffb99fp8,       0x1.0f41a1a5d8764p687},
            {0x1.dd3ffffffb99fp8,       0x1.70acf6623ff32p688},
            {0x1.de3ffffffb99fp8,       0x1.f515070ef61acp689},
            {0x1.df3ffffffb99fp8,       0x1.5485473c56dfcp691},
            {0x1.e03ffffffb99fp8,       0x1.ced0c4e4d59e4p692},
            {0x1.e13ffffffb99fp8,       0x1.3a8407ca209c6p694},
            {0x1.e23ffffffb99fp8,       0x1.ab78a196b76fdp695},
            {0x1.e33ffffffb99fp8,       0x1.227f2ee6fa305p697},
            {0x1.e43ffffffb99fp8,       0x1.8ad37b3bad33ep698},
            {0x1.e53ffffffb99fp8,       0x1.0c4fe93ccdf88p700},
            {0x1.e63ffffffb99fp8,       0x1.6cac8cde514efp701},
            {0x1.e73ffffffb99fp8,       0x1.efa4b5032b54fp702},
            {0x1.e83ffffffb99fp8,       0x1.50d31930266e8p704},
            {0x1.e93ffffffb99fp8,       0x1.c9cac6ffa71cbp705},
            {0x1.ea3ffffffb99fp8,       0x1.371a1bc09599dp707},
            {0x1.eb3ffffffb99fp8,       0x1.a6d4d9189f018p708},
            {0x1.ec3ffffffb99fp8,       0x1.1f580038307ccp710},
            {0x1.ed3ffffffb99fp8,       0x1.868a6825185b7p711},
            {0x1.ee3ffffffb99fp8,       0x1.09665f229766bp713},
            {0x1.ef3ffffffb99fp8,       0x1.68b741d009e02p714},
            {0x1.f03ffffffb99fp8,       0x1.ea437fb4bc319p715},
            {0x1.f13ffffffb99fp8,       0x1.4d2b3038bf046p717},
            {0x1.f23ffffffb99ep8,       0x1.c4d2be652cea4p718},
            {0x1.f33ffffffb99ep8,       0x1.33b9ac04c0212p720},
            {0x1.f43ffffffb99ep8,       0x1.a23df502a4a3ap721},
            {0x1.f53ffffffb99ep8,       0x1.1c39946547606p723},
            {0x1.f63ffffffb99ep8,       0x1.824d3d6b0103dp724},
            {0x1.f73ffffffb99ep8,       0x1.0684ec9be79c4p726},
            {0x1.f83ffffffb99ep8,       0x1.64ccf65229834p727},
            {0x1.f93ffffffb99ep8,       0x1.e4f13d25df78ap728},
            {0x1.fa3ffffffb99ep8,       0x1.498d6fcce3f5ep730},
            {0x1.fb3ffffffb99ep8,       0x1.bfe8844c8ea89p731},
            {0x1.fc3ffffffb99ep8,       0x1.30629e3b46a18p733},
            {0x1.fd3ffffffb99ep8,       0x1.9db3d1822eed1p734},
            {0x1.fe3ffffffb99ep8,       0x1.1923d3162d5cep736},
            {0x1.ff3ffffffb99ep8,       0x1.7e1bd9f724dbcp737},
            {0x1.001ffffffdccfp9,       0x1.03ab7b2c9b517p739},
            {0x1.009ffffffdccfp9,       0x1.60ed8bd54933ep740},
            {0x1.011ffffffdccfp9,       0x1.dfadc3cd79f32p741},
            {0x1.019ffffffdccfp9,       0x1.45f9bbb2a35d1p743},
            {0x1.021ffffffdccfp9,       0x1.bb0bf258b833ap744},
            {0x1.029ffffffdccfp9,       0x1.2d14d8520baf7p746},
            {0x1.031ffffffdccfp9,       0x1.99364b282dd34p747},
            {0x1.039ffffffdccfp9,       0x1.1616a4367556fp749},
            {0x1.041ffffffdccfp9,       0x1.79f61d0f30fbp750},
            {0x1.049ffffffdccfp9,       0x1.00d9f49709365p752},
            {0x1.051ffffffdccfp9,       0x1.5d18e41eebfc4p753},
            {0x1.059ffffffdccfp9,       0x1.da78ea95d97f9p754},
            {0x1.061ffffffdccfp9,       0x1.426ff7fe794b6p756},
            {0x1.069ffffffdccfp9,       0x1.b63ce2972ea1ep757},
            {0x1.071ffffffdccfp9,       0x1.29d0407f62d33p759},
            {0x1.079ffffffdccfp9,       0x1.94c53ee806803p760},
            {0x1.081ffffffdccfp9,       0x1.1311eff49ae9ap762},
            {0x1.089ffffffdccfp9,       0x1.75dbe653c2ceap763},
            {0x1.091ffffffdccfp9,       0x1.fc2085b6a9375p764},
            {0x1.099ffffffdccfp9,       0x1.594ee148935c8p766},
            {0x1.0a1ffffffdccfp9,       0x1.d55288db74dc3p767},
            {0x1.0a9ffffffdccfp9,       0x1.3ef00912762eap769},
            {0x1.0b1ffffffdccfp9,       0x1.b17b2f7ee7f4ep770},
            {0x1.0b9ffffffdccfp9,       0x1.2694bd41472c4p772},
            {0x1.0c1ffffffdccfp9,       0x1.90608a1681a3ep773},
            {0x1.0c9ffffffdccfp9,       0x1.10159ec1486a4p775},
            {0x1.0d1ffffffdccfp9,       0x1.71cd15bf6b516p776},
            {0x1.0d9ffffffdccfp9,       0x1.f69ca06d83551p777},
            {0x1.0e1ffffffdccfp9,       0x1.558f65bed5d1ap779},
            {0x1.0e9ffffffdccfp9,       0x1.d03a766bae541p780},
            {0x1.0f1ffffffdccfp9,       0x1.3b79d39d672c6p782},
            {0x1.0f9ffffffdccfp9,       0x1.acc6b3ef261e7p783},
            {0x1.101ffffffdccfp9,       0x1.2362355c94574p785},
            {0x1.109ffffffdccfp9,       0x1.8c080a68bcd97p786},
            {0x1.111ffffffdccfp9,       0x1.0d21994e9f02ap788},
            {0x1.119ffffffdccfp9,       0x1.6dc98ba5b51bp789},
            {0x1.121ffffffdccfp9,       0x1.f1280e4617147p790},
            {0x1.129ffffffdccfp9,       0x1.51da544077fa2p792},
            {0x1.131ffffffdccfp9,       0x1.cb308b8399fb7p793},
            {0x1.139ffffffdccfp9,       0x1.380d3c9a00e7ep795},
            {0x1.141ffffffdccfp9,       0x1.a81f4b2e55334p796},
            {0x1.149ffffffdccfp9,       0x1.20388fdc417d5p798},
            {0x1.151ffffffdccfp9,       0x1.87bb9df31efbp799},
            {0x1.159ffffffdccfp9,       0x1.0a35c88f80cbp801},
            {0x1.161ffffffdccfp9,       0x1.69d128b22d25ap802},
            {0x1.169ffffffdccfp9,       0x1.ebc2a4ab78bb5p803},
            {0x1.171ffffffdccfp9,       0x1.4e2f8fdd883d6p805},
            {0x1.179ffffffdccfp9,       0x1.c634a0cec7546p806},
            {0x1.181ffffffdccfp9,       0x1.34aa294e0c99ep808},
            {0x1.189ffffffdccfp9,       0x1.a384d0e8ecc3p809},
            {0x1.191ffffffdccfp9,       0x1.1d17b4109e865p811},
            {0x1.199ffffffdccfp9,       0x1.837b23284f5e9p812},
            {0x1.1a1ffffffdccfp9,       0x1.075215b6dcdc8p814},
            {0x1.1a9ffffffdccfp9,       0x1.65e3cde76e421p815},
            {0x1.1b1ffffffdccfp9,       0x1.e66c397f0e285p816},
            {0x1.1b9ffffffdccfp9,       0x1.4a8efbf67ceccp818},
            {0x1.1c1ffffffdccfp9,       0x1.c1468f660e51bp819},
            {0x1.1c9ffffffdccfp9,       0x1.31507f4997713p821},
            {0x1.1d1ffffffdccfp9,       0x1.9ef72130544cep822},
            {0x1.1d9ffffffdccfp9,       0x1.19ff898e9368dp824},
            {0x1.1e1ffffffdccfp9,       0x1.7f4678d82ff18p825},
            {0x1.1e9ffffffdccfp9,       0x1.04766a36fd582p827},
            {0x1.1f1ffffffdccfp9,       0x1.62015c9e2f34p828},
            {0x1.1f9ffffffdccfp9,       0x1.e124a317460fp829},
            {0x1.201ffffffdccfp9,       0x1.46f87c3b54d8ep831},
            {0x1.209ffffffdccfp9,       0x1.bc6630ce5faecp832},
            {0x1.211ffffffdccfp9,       0x1.2e00246624382p834},
            {0x1.219ffffffdccfp9,       0x1.9a761879cac82p835},
            {0x1.221ffffffdccfp9,       0x1.16eff82ee1911p837},
            {0x1.229ffffffdccfp9,       0x1.7b1d7e2eda2c6p838},
            {0x1.231ffffffdccfp9,       0x1.01a2afc0d75c2p840},
            {0x1.239ffffffdccfp9,       0x1.5e29b684536ep841},
            {0x1.241ffffffdccfp9,       0x1.dbebb83e52c5dp842},
            {0x1.249ffffffdccfp9,       0x1.436bf4aaba53p844},
            {0x1.251ffffffdccfp9,       0x1.b7935ef798935p845},
            {0x1.259ffffffdccfp9,       0x1.2ab8fec5df362p847},
            {0x1.261ffffffdccfp9,       0x1.9601939d5137dp848},
            {0x1.269ffffffdccfp9,       0x1.13e8e80d6759dp850},
            {0x1.271ffffffdccfp9,       0x1.770012b39ed67p851},
            {0x1.279ffffffdccfp9,       0x1.fdada086b9c52p852},
            {0x1.281ffffffdccfp9,       0x1.5a5cbd9bfe665p854},
            {0x1.289ffffffdccfp9,       0x1.d6c15030e89c1p855},
            {0x1.291ffffffdccfp9,       0x1.3fe9499128953p857},
            {0x1.299ffffffdccfp9,       0x1.b2cdf43b59797p858},
            {0x1.2a1ffffffdccfp9,       0x1.277af4d2d449ap860},
            {0x1.2a9ffffffdccfp9,       0x1.91996fd498417p861},
            {0x1.2b1ffffffdccfp9,       0x1.10ea41886590dp863},
            {0x1.2b9ffffffdccfp9,       0x1.72ee16480890ep864},
            {0x1.2c1ffffffdccfp9,       0x1.f8256bd5a1072p865},
            {0x1.2c9ffffffdccfp9,       0x1.569a543aa97d8p867},
            {0x1.2d1ffffffdccfp9,       0x1.d1a5429cffadp868},
            {0x1.2d9ffffffdccfp9,       0x1.3c705f8813894p870},
            {0x1.2e1ffffffdccfp9,       0x1.ae15cb5be04dap871},
            {0x1.2e9ffffffdccfp9,       0x1.2445ed3e27324p873},
            {0x1.2f1ffffffdccfp9,       0x1.8d3d8ab9f0bdep874},
            {0x1.2f9ffffffdccfp9,       0x1.0df3ed3fc701dp876},
            {0x1.301ffffffdccfp9,       0x1.6ee76926e122bp877},
            {0x1.309ffffffdccfp9,       0x1.f2ac9640362cap878},
            {0x1.311ffffffdccfp9,       0x1.52e25d083c6f6p880},
            {0x1.319ffffffdccfp9,       0x1.cc9767a099276p881},
            {0x1.321ffffffdccfp9,       0x1.39011b7511e91p883},
            {0x1.329ffffffdccfp9,       0x1.a96abf82e5bedp884},
            {0x1.331ffffffdccfp9,       0x1.2119ceff4e062p886},
            {0x1.339ffffffdccfp9,       0x1.88edc2473f3cfp887},
            {0x1.341ffffffdccfp9,       0x1.0b05d4146a019p889},
            {0x1.349ffffffdccfp9,       0x1.6aebebe3397ccp890},
            {0x1.351ffffffdccfp9,       0x1.ed42f5104656ep891},
            {0x1.359ffffffdccfp9,       0x1.4f34bafe2847bp893},
            {0x1.361ffffffdccfp9,       0x1.c79797c887ff8p894},
            {0x1.369ffffffdccfp9,       0x1.359b628909b0ap896},
            {0x1.371ffffffdccfp9,       0x1.a4ccac407db6ep897},
            {0x1.379ffffffdccfp9,       0x1.1df681534dcc2p899},
            {0x1.381ffffffdccfp9,       0x1.84a9f4d4f272ap900},
            {0x1.389ffffffdccfp9,       0x1.081fdf276bf44p902},
            {0x1.391ffffffdccfp9,       0x1.66fb7f67746f6p903},
            {0x1.399ffffffdccfp9,       0x1.e7e85e064cb73p904},
            {0x1.3a1ffffffdccfp9,       0x1.4b91516684d28p906},
            {0x1.3a9ffffffdccfp9,       0x1.c2a5ac0f3d022p907},
            {0x1.3b1ffffffdccfp9,       0x1.323f1a3f5edc4p909},
            {0x1.3b9ffffffdccfp9,       0x1.a03b6d89faf11p910},
            {0x1.3c1ffffffdccfp9,       0x1.1adbebbbf9351p912},
            {0x1.3c9ffffffdccfp9,       0x1.80720118fc863p913},
            {0x1.3d1ffffffdccfp9,       0x1.0541f7d976c95p915},
            {0x1.3d9ffffffdccfp9,       0x1.631604f454091p916},
            {0x1.3e1ffffffdccfp9,       0x1.e29ca75828ca4p917},
            {0x1.3e9ffffffdccfp9,       0x1.47f803db3082ep919},
            {0x1.3f1ffffffdccfp9,       0x1.bdc17ddb963f2p920},
            {0x1.3f9ffffffdccfp9,       0x1.2eec285d246c2p922},
            {0x1.401ffffffdccfp9,       0x1.9bb6dfb8d5a7bp923},
            {0x1.409ffffffdccfp9,       0x1.17c9f5ff316ddp925},
            {0x1.411ffffffdccfp9,       0x1.7c45c625cf3bcp926},
            {0x1.419ffffffdccfp9,       0x1.026c07ca1067p928},
            {0x1.421ffffffdccfp9,       0x1.5f3b5e200997dp929},
            {0x1.429ffffffdccfp9,       0x1.dd5fa7afd8289p930},
            {0x1.431ffffffdccfp9,       0x1.4468b644f2c7fp932},
            {0x1.439ffffffdccfp9,       0x1.b8eae6ffb1c26p933},
            {0x1.441ffffffdccfp9,       0x1.2ba272f04fa72p935},
            {0x1.449ffffffdccfp9,       0x1.973edf89954efp936},
            {0x1.451ffffffdccfp9,       0x1.14c088262904bp938},
            {0x1.459ffffffdccfp9,       0x1.782523695aef2p939},
            {0x1.461ffffffdccfp9,       0x1.ff3bf1add8023p940},
            {0x1.469ffffffdccfp9,       0x1.5b6b6cd54844dp942},
            {0x1.471ffffffdccfp9,       0x1.d831362a33e02p943},
            {0x1.479ffffffdccfp9,       0x1.40e34cdaa0c9dp945},
            {0x1.481ffffffdccfp9,       0x1.b421c1b7c391cp946},
            {0x1.489ffffffdccfp9,       0x1.2861e04eed965p948},
            {0x1.491ffffffdccfp9,       0x1.92d34a1abd55p949},
            {0x1.499ffffffdccfp9,       0x1.11bf8a7ca8dc9p951},
            {0x1.4a1ffffffdccfp9,       0x1.740ff8ac10594p952},
            {0x1.4a9ffffffdccfp9,       0x1.f9af6a367aa8cp953},
            {0x1.4b1ffffffdccfp9,       0x1.57a613525a444p955},
            {0x1.4b9ffffffdccfp9,       0x1.d3112a55b14f7p956},
            {0x1.4c1ffffffdccfp9,       0x1.3d67ac204487bp958},
            {0x1.4c9ffffffdccfp9,       0x1.af65e8a8eee76p959},
            {0x1.4d1ffffffdccfp9,       0x1.252a57165ab35p961},
            {0x1.4d9ffffffdccfp9,       0x1.8e73fcebbcdf2p962},
            {0x1.4e1ffffffdccfp9,       0x1.0ec6e5905727p964},
            {0x1.4e9ffffffdccfp9,       0x1.7006260fe518p965},
            {0x1.4f1ffffffdccfp9,       0x1.f4324dde1b853p966},
            {0x1.4f9ffffffdccfp9,       0x1.53eb342838912p968},
            {0x1.501ffffffdccfp9,       0x1.cdff5c312677p969},
            {0x1.509ffffffdccfp9,       0x1.39f5b8e646512p971},
            {0x1.511ffffffdccfp9,       0x1.aab736e0229e8p972},
            {0x1.519ffffffdccfp9,       0x1.21fbbe2a7cc33p974},
            {0x1.521ffffffdccfp9,       0x1.8a20d5dbe17b1p975},
            {0x1.529ffffffdccfp9,       0x1.0bd6823000616p977},
            {0x1.531ffffffdccfp9,       0x1.6c078c0f5af0cp978},
            {0x1.539ffffffdccfp9,       0x1.eec471cd26923p979},
            {0x1.541ffffffdccfp9,       0x1.503ab239a52eep981},
            {0x1.549ffffffdccfp9,       0x1.c8fba42a91b8p982},
            {0x1.551ffffffdccfp9,       0x1.368d584898928p984},
            {0x1.559ffffffdccfp9,       0x1.a61587d0f8cb4p985},
            {0x1.561ffffffdccfp9,       0x1.1ed5fcb4fed8p987},
            {0x1.569ffffffdccfp9,       0x1.85d9b3294cbccp988},
            {0x1.571ffffffdccfp9,       0x1.08ee496ae24e1p990},
            {0x1.579ffffffdccfp9,       0x1.68140b7c89c5cp991},
            {0x1.581ffffffdccfp9,       0x1.e965aba3129a2p992},
            {0x1.589ffffffdccfp9,       0x1.4c9470ba47e91p994},
            {0x1.591ffffffdccfp9,       0x1.c405db1de4f69p995},
            {0x1.599ffffffdccfp9,       0x1.332e6fade5fp997},
            {0x1.5a1ffffffdccfp9,       0x1.a180b7549973p998},
            {0x1.5a9ffffffdccfp9,       0x1.1bb8fa248f731p1000},
            {0x1.5b1ffffffdccfp9,       0x1.819e736fecc0ap1001},
            {0x1.5b9ffffffdccfp9,       0x1.060e248ff8e45p1003},
            {0x1.5c1ffffffdccfp9,       0x1.642b85802c37dp1004},
            {0x1.5c9ffffffdccfp9,       0x1.e415d17516714p1005},
            {0x1.5d1ffffffdccfp9,       0x1.48f8532dcd8aap1007},
            {0x1.5d9ffffffdccfp9,       0x1.bf1dda53d4168p1008},
            {0x1.5e1ffffffdccfp9,       0x1.2fd8e4c6c1a74p1010},
            {0x1.5e9ffffffdccfp9,       0x1.9cf8a1a8a05e1p1011},
            {0x1.5f1ffffffdccfp9,       0x1.18a49e2c20c14p1013},
            {0x1.5f9ffffffdccfp9,       0x1.7d6ef5a877896p1014},
            {0x1.601ffffffdccfp9,       0x1.0335fd2d4d32ap1016},
            {0x1.609ffffffdccfp9,       0x1.604ddb98aeeb6p1017},
            {0x1.611ffffffdccfp9,       0x1.ded4b9cce1c65p1018},
            {0x1.619ffffffdccep9,       0x1.45663d67095d1p1020},
            {0x1.621ffffffdccep9,       0x1.ba437b80a6915p1021},
            {0x1.629ffffffdccep9,       0x1.2c8c9d8cda0c8p1023},
        };

        for (double[] testCase: testCases)
            failures+=testExpm1Case(testCase[0], testCase[1]);

        return failures;
    }
}
